﻿using System;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace PWS.API.Graphic.Paint
{
    public partial class FormMain : Form
    {
        #region Fields (5)

        public Color backColor;
        //******
        Graphics graphicBoard;
        public IntPtr hDC;
        Rectangle rect;
        public int shapeId = 0, paperX, paperY, startX, startY, endX, endY;

        #endregion Fields

        #region Enums (2)

        public enum TernaryRasterOperations : uint
        {
            SrcCopy = 0x00CC0020,
        }

        [Flags]
        public enum PenStyle
        {
            PS_Solid = 0,
            PS_Dash = 1,
            PS_Dot = 2,
            PS_DashDot = 3,
            PS_DashDotDot = 4,
            PS_Null = 5,
            PS_InSideFrame = 6,
            PS_UserStyle = 7,
            PS_Alternate = 8,
            PS_StyleMask = 0x0000000F,

            PS_EndCapRound = 0x00000000,
            PS_EndCapSquare = 0x00000100,
            PS_EndCapFlat = 0x00000200,
            PS_EndCapMask = 0x00000F00,

            PS_JoinRound = 0x00000000,
            PS_JoinBevel = 0x00001000,
            PS_JoinMiter = 0x00002000,
            PS_JoinMask = 0x0000F000,

            PS_CosMetric = 0x00000000,
            PS_GeoMetric = 0x00010000,
            PS_TypeMask = 0x000F0000
        }

        #endregion Enums

        #region Constructors (1)

        public FormMain()
        {
            InitializeComponent();
        }

        #endregion Constructors

        #region Methods (42)

        // Public Methods (14) 

        [DllImport("gdi32.dll")]
        public static extern bool Arc(IntPtr hDC, int nLeftRect, int nTopRect, int nRightRect, int nBottomRect, int nXStartArc, int nYStartArc, int nXEndArc, int nYEndArc);

        public int ColorToRGB(Color crColor)
        {
            return crColor.B << 16 | crColor.G << 8 | crColor.R;
        }

        [DllImport("gdi32.dll")]
        public static extern IntPtr CreatePen(PenStyle penStyle, int nWidth, int penColor);

        [DllImport("gdi32.dll")]
        public static extern bool Ellipse(IntPtr hDC, int nLeftRect, int nTopRect, int nRightRect, int nBottomRect);

        [DllImport("user32.dll")]
        public static extern bool GetCursorPos(ref Point lpPoint);

        [DllImport("user32.dll")]
        public static extern IntPtr GetDC(IntPtr hWnd);

        [DllImport("user32.dll")]
        public static extern bool InvertRect(IntPtr hDC, [In] ref Rectangle lpRect);

        [DllImport("gdi32.dll")]
        public static extern bool LineTo(IntPtr hDC, int nXEnd, int nYEnd);

        [DllImport("gdi32.dll")]
        public static extern bool MoveToEx(IntPtr hDC, int x, int y, IntPtr lpPoint);

        [DllImport("gdi32.dll")]
        public static extern bool Polygon(IntPtr hDC, Point[] lpPoints, int pointsCount);

        [DllImport("gdi32.dll")]
        public static extern bool Rectangle(IntPtr hDC, int nLeftRect, int nTopRect, int nRightRect, int nBottomRect);

        [DllImport("gdi32.dll")]
        public static extern bool RoundRect(IntPtr hDC, int nLeftRect, int nTopRect, int nRightRect, int nBottomRect, int nWidth, int nHeight);

        [DllImport("gdi32.dll")]
        public static extern IntPtr SelectObject(IntPtr hDC, IntPtr hObject);

        [DllImport("gdi32.dll")]
        public static extern bool SetPixel(IntPtr hDC, int x, int y, int crColor);

        // Private Methods (28) 

        [DllImport("gdi32.dll")]
        private static extern bool BitBlt(IntPtr hDC, int nXDest, int nYDest, int nWidth, int nHeight, IntPtr hdcSrc, int nXSrc, int nYSrc, TernaryRasterOperations dwRop);

        private void BkColorBtnClick(object sender, EventArgs e)
        {
            if (colorDialogMain.ShowDialog() == DialogResult.OK)
                backColor = panelDraw.BackColor = BkColorBTN.BackColor = colorDialogMain.Color;
        }

        [DllImport("gdi32.dll")]
        private static extern bool Chord(IntPtr hdc, int nLeftRect, int nTopRect, int nRightRect, int nBottomRect, int nXRadial1, int nYRadial1, int nXRadial2, int nYRadial2);

        private void ClearBtnClick(object sender, EventArgs e)
        {
            graphicBoard.Clear(backColor);
        }

        private void ClearToolStripMenuItemClick(object sender, EventArgs e)
        {
            ClearBtnClick(null, null);
        }

        private void ColorBtnClick(object sender, EventArgs e)
        {
            colorDialogMain.ShowDialog();
            ColorBTN.BackColor = colorDialogMain.Color;

            IntPtr pen = CreatePen(PenStyle.PS_Solid, (int)PenWidth.Value, ColorToRGB(ColorBTN.BackColor));
            SelectObject(hDC, pen);
        }

        private void Draw()
        {
            switch (shapeId)
            {
                case 1: Rectangle(hDC, startX, startY, endX, endY); break;

                case 2: SetPixel(hDC, endX, endY, ColorToRGB(ColorBTN.BackColor)); break;

                case 3:
                    {
                        int r = ((endX - startX) + (endY - startY)) / 2;
                        Ellipse(hDC, startX, startY, startX + r, startY + r);
                    }
                    break;

                case 4: Ellipse(hDC, startX, startY, endX, endY); break;

                case 5: Arc(hDC, startX, startY, endX, endY, startX, startY, endX, endY); break;

                case 6: RoundRect(hDC, startX, startY, endX, endY, 10, 10); break;

                case 7:
                    {
                        MoveToEx(hDC, startX, startY, IntPtr.Zero);
                        LineTo(hDC, endX, endY);
                    }
                    break;

                case 8:
                    {
                        Point[] trianglePoints =
                        {
                            new Point (startX + (endX - startX) /2 ,startY),new Point (endX,endY),new Point (startX,endY)
                        };

                        Polygon(hDC, trianglePoints, 3);
                    } break;

                case 9: Chord(hDC, startX, startY, endX, endY, startX, startY, endX, endY); break;

                case 10: BitBlt(hDC, 1, 1, endX - startX, endY - startY, hDC, startX, startY, TernaryRasterOperations.SrcCopy); break;

                case 11: InvertRect(hDC, ref rect); break;
            }
        }

        private void ExitToolStripMenuItemClick(object sender, EventArgs e)
        {
            Close();
        }

        private void FormMain_Load(object sender, EventArgs e)
        {
            graphicBoard = panelDraw.CreateGraphics();
            hDC = GetDC(panelDraw.Handle);
            backColor = panelDraw.BackColor;
        }

        private void NewToolStripMenuItemClick(object sender, EventArgs e)
        {
            graphicBoard.Clear(Color.White);
            backColor = BkColorBTN.BackColor = Color.White;
            ColorBTN.BackColor = Color.Black;
            IntPtr pen = CreatePen(PenStyle.PS_Solid, 1, ColorToRGB(Color.Black));
            SelectObject(hDC, pen);
            PenWidth.Value = 1;
            pentool.Checked = true;
        }

        private void PanelDraw_MouseDown(object sender, MouseEventArgs e)
        {
            rect.Width = startX = paperX;
            rect.Height = startY = paperY;
            if (shapeId == 0) timerPencil.Enabled = true;
            MoveToEx(hDC, paperX, paperY, IntPtr.Zero);
        }

        private void PanelDraw_MouseLeave(object sender, EventArgs e)
        {
            timerPencil.Enabled = false;
            PositionLabel.Text = @"X: 0\nY: 0";
        }

        private void PanelDraw_MouseMove(object sender, MouseEventArgs e)
        {
            Point pt = new Point(0, 0);
            if (!GetCursorPos(ref pt)) return;
            paperX = pt.X - this.Location.X - 6;
            paperY = pt.Y - this.Location.Y - 95;
            PositionLabel.Text = String.Format("X: {0}\nY: {1}", paperX, paperY);
        }

        private void PanelDraw_MouseUp(object sender, MouseEventArgs e)
        {
            rect.Width = endX = paperX;
            rect.Height = endY = paperY;
            timerPencil.Enabled = false;
            Draw();
        }

        private void PenWidth_ValueChanged(object sender, EventArgs e)
        {
            IntPtr pen = CreatePen(PenStyle.PS_Solid, (int)PenWidth.Value, ColorToRGB(ColorBTN.BackColor));
            SelectObject(hDC, pen);
        }

        private void RadioArcTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 5;
        }

        private void RadioChordTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 9;
        }

        private void RadioCircleTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 3;
        }

        private void RadioCopyTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 10;
        }

        private void RadioDotTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 2;
        }

        private void RadioEllipseTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 4;
        }

        private void RadioInverseTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 11;
        }

        private void RadioLineTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 7;
        }

        private void RadioLineToTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 8;
        }

        private void RadioPenTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 0;
        }

        private void RadioRectangleTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 1;
        }

        private void RadioRoundRectTool_CheckedChanged(object sender, EventArgs e)
        {
            shapeId = 6;
        }

        private void TimerPencil_Tick(object sender, EventArgs e)
        {
            LineTo(hDC, paperX, paperY);
        }

        #endregion Methods
    }
}